#ifndef __msg_queues_h
#define __msg_queues_h

// The maximum length (in bytes) of a message
#define MSGLEN 254

// The maximum number of messages in a single queue
#define MSGQUEUELEN 2   // DO NOT CHANGE THIS!

typedef struct __msg {
    unsigned char full; // Indicates that data resides in this msg
    unsigned char length; // Length of message data to store
    unsigned char msgtype; // Indicates type of message (see maindefs.h)

    unsigned char *data; // The actual message data
} msg;  // 5 bytes overhead per msg

typedef struct __msg_queue {
    msg queue[MSGQUEUELEN];
    unsigned char cur_write_index; // Location to write to next
    unsigned char cur_read_index; // Location to read from next
} msg_queue;    // 12 bytes overhead per queue

// Error Codes
#define MSG_SEND_OKAY 1         // Message sent okay
#define MSG_QUEUE_FULL -1       // Too many messages in the queue
#define MSG_BAD_LEN -2          // The length of the message is either too large or negative
#define MSG_BUFFER_TOOSMALL -3  // The message buffer is too small to receive the message in the queue
#define MSG_QUEUE_EMPTY -4      // The message queue is empty
#define MSG_NOT_IN_LOW -5       // This call must be made from a low-priority interrupt handler
#define MSG_NOT_IN_HIGH -6      // This call must be made from a high-priority interrupt handler
#define MSG_NOT_IN_MAIN -7      // This call must be made from the "main()" thread

// This MUST be called before anything else in messages and should
// be called before interrupts are enabled
void MQ_init(void);

// This is called from a high priority interrupt to decide if the
// processor may sleep. It is currently called in interrupts.c
void MQ_sleep_high_interrupt_if_okay(void);

// This is called in the "main()" thread (if desired) to block
// until a message is received on one of the two incoming queues
void MQ_wait_on_incoming_msg_queues(void);

// Queue:
// The "MQ_ToMainToLow" queue is a message queue from low priority
//  interrupt handlers to the "main()" thread.  The send is called
//  in the interrupt handlers and the receive from "main()"
signed int	MQ_sendmsg_ToMainFromLow(unsigned char,unsigned char,void *);
signed int	MQ_recvmsg_ToMainFromLow(unsigned char,unsigned char *,void *);
unsigned int   MQ_peek_ToMainFromLow(void);

// Queue:
// The "MQ_ToMainFromHigh" queue is a message queue from high priority
//  interrupt handlers to the "main()" thread.  The send is called
//  in the interrupt handlers and the receive from "main()"
signed int	MQ_sendmsg_ToMainFromHigh(unsigned char,unsigned char,void *);
signed int	MQ_recvmsg_ToMainFromHigh(unsigned char,unsigned char *,void *);
unsigned int   MQ_peek_ToMainFromHigh(void);

// Queue:
// The "MQ_FromMainToLow" queue is a message queue from the "main()"
//  thread to the low priority interrupt handlers.  The send is called
//  in the "main()" thread and the receive from the interrupt handlers.
signed int	MQ_sendmsg_FromMainToLow(unsigned char,unsigned char,void *);
signed int	MQ_recvmsg_FromMainToLow(unsigned char,unsigned char *,void *);
unsigned int   MQ_peek_FromMainToLow(void);

// Queue:
// The "MQ_FromMainToHigh" queue is a message queue from the "main()"
//  thread to the high priority interrupt handlers.  The send is called
//  in the "main()" thread and the receive from the interrupt handlers.
signed int	MQ_sendmsg_FromMainToHigh(unsigned char,unsigned char,void *);
signed int	MQ_recvmsg_FromMainToHigh(unsigned char,unsigned char *,void *);
unsigned int   MQ_peek_FromMainToHigh(void);

#endif
